% --- Атомарные знания ---
person(p1, 60_000).
child(p1).
vehicle(v1, motorbike, 60_000).
animal(a1, dog, 60_000).
bag(b1, 60_000).
window(s1, 60_000).
door(d1, 60_000).
% Позция объекта в определенный момент времени (x, y, z). Присутствует для каждого объекта, у которого
% также есть volumetricPosition.
position(any_object, point(x, y, z), 60_000).
% Позиция, задаваемая в виде многоугольника. Многоугольник представлен набором точек, лежащих на поверхности, 
volumetricPosition(obj, [point(x1, y1, z1), point(x2, y2, z2)], 60_000).
% Вектор взгляда человека, проецированный на плоскость земли. Определяется исходя из поворота головы человека.
lookDirection(p1, vector(vx, vy, vz), 60_000).
% Двумерное расположение объекта в виде обрамляющей рамка.
bounds(any_obj, box(x1, y1, x2, y2), 60_000).
started(sit_down(p1), 61_000).
started(search(p1, p2), 63_000).
finished(sit_down(p1), 62_000).
finished(search(p1, p2), 64_000).
% Устанавливается один раз - используемый временной шаг
timeStep(1000).
% Время кадра, информация о котором присутствует в базе знаний
timeSlice(60_000).
% Описываем все точки, относящиеся к данному положению в метрах. Гарантируется, что координаты x, y в position
% будут соответствовать - будут заадваться точно для пикселя.
areaLocation(topLeftRoadArea, x, y).
direction(komendanskiyProsperctEast, topLeftRoadArea, vx, vy).

areaLocation(mainHighwayArea, x, y).
highway(mainHighway, mainHighwayArea).

% --- Utility ---

% В случае, если MaxTime - переменная, предикат вернет наибольшее время в базе знаний.
% Опирается на гарантию того, что новые знания зыписываются в начало.
maxTimeSlice(MaxTime) :-
	timeSlice(MaxTime),
	!.

toMillis(dur(Dur, millis), DurMillis) :-
	number(Dur),
	DurMillis is Dur.
toMillis(dur(Dur, sec), DurMillis) :-
	number(Dur),
	DurMillis is Dur * 1000.
toMillis(dur(Dur, min), DurMillis) :-
	number(Dur),
	DurMillis is Dur * 60_000.
toMillis(DurMillis, DurMillis) :-
	number(DurMillis).

% Преобразование расстояния в миллиметры
toMm(dist(Dist, mm), DistMm) :-
	number(Dist),
	DistMm is Dist.
toMm(dist(Dist, m), DistMm) :-
	number(Dist),
	DistMm is Dist * 1000.
toMm(DistMm, DistMm) :-
	number(DistMm).

incTime(BaseTime, Dur, ResultTime) :-
	toMillis(Dur, DurMs),
	ResultTime is BaseTime + DurMs.

decTime(BaseTime, Dur, ResultTime) :-
	toMillis(Dur, DurMs),
	ResultTime is BaseTime - DurMs.

% Итерирует от MinIter до MaxIter включительно с шагом Step.
iter(MinIter, _Step, MaxIter, _Iter) :-
	MinIter > MaxIter,
	!, fail.
iter(MinIter, _Step, _MaxIter, MinIter).
iter(MinIter, Step, MaxIter, Iter) :-
	NextMinIter is MinIter + Step,
	iter(NextMinIter, Step, MaxIter, Iter).

% Задает время, большее StartTime, но не более чем на MaxTimeInt
timeGreater(MinTime, MaxTimeDifference, GreaterTime) :-
	timeSlice(MinTime),
	timeStep(TimeStep),
	toMillis(MaxTimeDifference, MaxTimeDifferenceMillis),
	iter(TimeStep, TimeStep, MaxTimeDifferenceMillis, TimeInt),
	GreaterTime is MinTime + TimeInt.

timeSmaller(MaxTime, MaxTimeDiff, SmallerTime) :-
	timeSlice(MaxTime),
	timeStep(TimeStep),
	toMillis(MaxTimeDiff, MaxTimeDifferenceMillis),
	iter(TimeStep, TimeStep, MaxTimeDifferenceMillis, TimeInt),
	SmallerTime is MaxTime - TimeInt,
	SmallerTime >= 0.

timeBetween(StartTime, FinishTime, Time) :-
	timeStep(TimeStep),
	timeSlice(StartTime),
	timeSlice(FinishTime),
	StartTime =< FinishTime,
	MaxTimeInt is FinishTime - StartTime,
	iter(0, TimeStep, MaxTimeInt, TimeInt),
	Time is StartTime + TimeInt.

% Гарантируется, что между started и finished не вклинится лишний started или finished. А также, в один
% момент времени действие не может закончиться и сразу начаться, но может начаться и сразу закончиться (в этом
% случае StarTime и FinishTime будут равны). Считаем, что утверждение выполнялось от StartTime до FinishTime
% (включительно оба времени).
holds(Statement, StartTime, FinishTime) :-
	started(Statement, StartTime),
	finished(Statement, FinishTime),
	StartTime =< FinishTime,
	\+ (finished(Statement, IntermedTime), StartTime < IntermedTime, IntermedTime < FinishTime).
holds(Statement, StartTime, FinishTime) :-
	started(Statement, StartTime),
	\+ (finished(Statement, FinishTime), StartTime =< FinishTime),
	maxTimeSlice(FinishTime).
holdsAt(Statement, Time) :-
	holds(Statement, StartTime, FinishTime),
	timeBetween(StartTime, FinishTime, Time).

% Предикаты для удобной работы с временем
holdsAt(person(Pers), Time) :-
	person(Pers, Time).

holdsAt(bag(Bag), Time) :-
	bag(Bag, Time).

holdsAt(vehicle(Veh, Type), Time) :-
	vehicle(Veh, Type, Time).

holdsAt(position(Obj, P), Time) :-
	position(Obj, P, Time).

% Предикаты для работы с точками и векторами
vectorOf(point(X1, Y1, Z1), point(X2, Y2, Z2), vector(VX, VY, VZ)) :-
	VX is X2-X1,
	VY is Y2-Y1,
	VZ is Z2-Z1.

vectorLength(vector(X, Y, Z), Length) :-
	Length is sqrt(X**2 + Y**2 + Z**2).

distanceBetweenPoints(P1, P2, Dist) :-
	vectorOf(P1, P2, V),
	vectorLength(V, Dist).

scalarMult(vector(X1, Y1, Z1), vector(X2, Y2, Z2), Scal) :-
	Scal is X1*X2 + Y1*Y2 + Z1*Z2.

cosBetween(V1, V2, Cos) :-
	scalarMult(V1, V2, Scal),
	vectorLength(V1, L1),
	vectorLength(V2, L2),
	Cos is Scal / (L1 * L2).

vectorMult(vector(A1, A2, A3), vector(B1, B2, B3), vector(S1, S2, S3)) :-
	S1 is A2*B3 - A3*B2,
	S2 is A3*B1 - A1*B3,
	S3 is A1*B2 - A2*B1.

pointBetween(point(X1, Y1, Z1), point(X2, Y2, Z2), point(X3, Y3, Z3)) :-
	X3 is (X1 + X2) / 2,
	Y3 is (Y1 + Y2) / 2,
	Z3 is (Z1 + Z2) / 2.

% Возвращает ближайшее время, меньшее Time, когда выполнялось указанное утверждение. Ожидается, что Statement и Time будут заданы.
% Оптимизирует поиск, предполагая, что утверждение выполнялось не более минуты назад.
holdsAtNearestPrevTime(Statement, Time, PrevTime) :-
	timeSmaller(Time, dur(1, min), PrevTime),
	holdsAt(Statement, PrevTime),
	% Поскольку новые события записываются в начало, останавливаемся на первом подходящем моменте времени.
	!.
holdsAtNearestPrevTime(Statement, Time, PrevTime) :-
	% Ожидает, чтоболее свежие знания записываются в начало
	holdsAt(Statement, PrevTime),
	PrevTime < Time,
	!.

% --- Иерархические связи между действиями ---
started(hitting(Person), Time) :-
	started(hittingWithHand(Person), Time).
started(hitting(Person), Time) :-
	started(kicking(Person), Time).

finished(hitting(Person), Time) :-
	finished(hitWithHand(Person), Time).
finished(hitting(Person), Time) :-
	started(kicking(Person), Time).

% --- Предикаты, описывающие сцену ---

% Средство передвижения подъехало к человеку (сместилось и находится на расстоянии не более 15 м от человека).
% При этом не обязательна остановка (злоумышленник мог высадить подельник на ходу).
% Правило универсально и для машины, и для скутера.
droveUpTo(Veh, Pers, StartTime, FinishTime) :-
	vehicle(Veh, _, FinishTime),
	person(Pers, FinishTime),
	distanceNotGreater(Pers, Veh, dist(15, meter), FinishTime),
	moved(Veh, dist(1, meter), dur(1, min), StartTime, FinishTime),
	\+ (timeBetween(StartTime, FinishTime, Time), onVehic(Veh, Pers, Time)).

% Средство передвижения отъехало от человека (находилось на расстоянии не более 15 м от человека и уехало).
droveAway(Veh, Pers, StartTime, FinishTime) :-
	vehicle(Veh, _, StartTime),
	person(Pers, StartTime),
	distanceNotGreater(Pers, Veh, dist(15, meter), StartTime),
	moved(Veh, dist(1, meter), dur(1, min), StartTime, FinishTime),
	\+ (timeBetween(StartTime, FinishTime, Time), onVehic(Veh, Pers, Time)).

% Obj переместился не менее чем на MinDist за не более чем MaxDur от времени StartTime до FinishTime.
moved(Ojb, MinDist, MaxDur, StartTime, FinishTime) :-
	toMm(MinDist, MinDistMm),
	timeSlice(FinishTime),
	movedMinInterval(Ojb, MinDistMm, MaxDur, StartTime, FinishTime).

% Ожидает, что FinishTime задан. Находит максимальный StartTime, подходящее под критерии moved.
movedMinInterval(Ojb, MinDistMm, MaxDur, StartTime, FinishTime) :-
	position(Ojb, P2, FinishTime),
	timeSmaller(FinishTime, MaxDur, StartTime),
	position(Ojb, P1, StartTime),
	distanceBetweenPoints(P1, P2, Dist),
	Dist >= MinDistMm,
	!.

% Проверяет, что расстояние между двумя точками не больше заданного ограничения.
distanceNotGreater(Obj1, Obj2, MaxDist, Time) :-
	toMm(MaxDist, MaxDistMm),
	distanceBetween(Obj1, Obj2, Time, Dist), 
	Dist =< MaxDistMm.

% Проверяет, что расстояние между двумя точками больше заданного ограничения.
distanceGreater(Obj1, Obj2, MinDist, Time) :-
	toMm(MinDist, MinDistMm),
	distanceBetween(Obj1, Obj2, Time, Dist),
	Dist > MinDistMm.

distanceNotGreaterHappensBetween(Obj1, Obj2, MaxDistance, StartTime, FinishTime) :-
	timeBetween(StartTime, FinishTime, Time),
    distanceNotGreater(Obj1, Obj2, MaxDistance, Time),
    !.

distanceBetween(Obj1, Obj2, Time, Dist) :-
	(person(Obj2, Time) ; animal(Obj2, _, Time)),
	position(Obj1, P1, Time),
	position(Obj2, P2, Time),
	Obj1 \= Obj2,
	distanceBetweenPoints(P1, P2, Dist).
distanceBetween(Obj1, Obj2, Time, Dist) :-
	(vehicle(Obj2, _, Time) ; window(Obj2, Time) ; door(Obj2, Time) ; fence(Obj2, Time) ; wall(Obj2, Time)),
	position(Obj1, P, Time),
	volumetricPosition(Obj2, [FirstP | TailP], Time),
	Obj1 \= Obj2,
	minPointToLineDistance(P, [FirstP | TailP], FirstP, 1_000_000_000, Dist).

minPointToLineDistance(P, [P1, P2 | TP], FirstP, MinDistAcc, MinDist) :-
	pointToLineDistance(P, P1, P2, Dist),
	NewMinDist is min(MinDistAcc, Dist),
	minPointToLineDistance(P, [P2 | TP], FirstP, NewMinDist, MinDist).
minPointToLineDistance(P, [LastP], FirstP, MinDistAcc, MinDist) :-
	pointToLineDistance(P, LastP, FirstP, Dist),
	MinDist is min(MinDistAcc, Dist).

pointToLineDistance(P, LP1, LP2, Dist) :-
	distanceBetweenPoints(P, LP1, AB),
	distanceBetweenPoints(P, LP2, AC),
	(AB < AC -> ClosestP = LP1, FarthestP = LP2 ; ClosestP = LP2, FarthestP = LP1),
	vectorOf(ClosestP, P, V1),
	vectorOf(ClosestP, FarthestP, V2),
	scalarMult(V1, V2, Scal),
	(Scal < 0 -> distanceBetweenPoints(P, ClosestP, Dist) ;
		distanceBetweenPoints(LP1, LP2, BC), vectorOf(P, LP1, ABVec),
		vectorOf(P, LP2, ACVec), cosBetween(ABVec, ACVec, Cos),
		Dist is AB * AC * sin(acos(Cos)) / BC).

% Человек покинул средство предвижения (мотоцикл, машину и т.п.).
% Для мотоцикла - если человек находился "на" мотоцикле.
% Считаем, что действие моментальное. Time - первый момент, когда стало ясно, что человек больше не
% на средстве передвижения.
leftAVehicle(Veh, Pers, LiveTime) :-
	(vehicle(Veh, motorbike, LiveTime); vehicle(Veh, bicycle, LiveTime)),
	person(Pers, LiveTime),
	\+ onVehic(Veh, Pers, LiveTime),
	holdsAtNearestPrevTime(person(Pers), LiveTime, PrevTime),
	onVehic(Veh, Pers, PrevTime).

% Считаем, что человек вышел из машины, если человек появился возле машины, в течение прошлых 10 секунд человек отсутствовал
% на видео, и в течение 5 секунд машина не ехала (это не проезжающая мимо машина).
leftAVehicle(Veh, Pers, LiveTime) :-
	vehicle(Veh, car, LiveTime),
	person(Pers, LiveTime),
	distanceNotGreater(Pers, Veh, dist(1.5, m), LiveTime),
	timeStep(TimeStep),
	CheckFinishTime is LiveTime - TimeStep, 
	decTime(CheckFinishTime, dur(10, sec), CheckStartTime),
	timeSlice(CheckStartTime),
	\+ (timeBetween(CheckStartTime, CheckFinishTime, CheckTime),
		person(Pers, CheckTime)),
	vehicleNotMoving(Veh, LiveTime).

% Считаем, что средство передвижения не двигается (возможно очень медленное движение), если в течение последних 5 секунд
% оно сместилось не более чем на 1 метр.
vehicleNotMoving(Veh, FinishTime) :-
	vehicle(Veh, _, FinishTime),
	position(Veh, P1, FinishTime),
	decTime(FinishTime, dur(5, sec), BaseStartTime),
	holdsAtNearestPrevTime(position(Veh, P2), BaseStartTime, _StartTime),
	distanceBetweenPoints(P1, P2, Dist),
	Dist < 1000.

% Человек сел на/в средство предвижения (мотоцикл, машину и т.п.).
jointAVehicle(Veh, Pers, JoinTime) :-
	(vehicle(Veh, motorbike, JoinTime); vehicle(Veh, bicycle, JoinTime)),
	person(Pers, JoinTime),
	onVehic(Veh, Pers, JoinTime),
	holdsAtNearestPrevTime(person(Pers), JoinTime, PrevTime),
	\+ onVehic(Veh, Pers, PrevTime).

% Считаем, что человек сел в машину, если человек стоял рядом с машиной, затем человек пропал из виду и в течение последующих
% 10 секунд отсутствовал на видео, и в течение 5 секунд до этого события машина не ехала.
jointAVehicle(Veh, Pers, JoinTime) :-
	vehicle(Veh, car, JoinTime),
	timeStep(TimeStep),
	PrevTime is JoinTime - TimeStep,
	person(Pers, PrevTime),
	distanceNotGreater(Pers, Veh, dist(1.5, m), PrevTime),
	CheckStartTime is JoinTime,
	incTime(CheckStartTime, dur(10, sec), CheckFinishTime),
	timeSlice(CheckFinishTime),
	\+ (timeBetween(CheckStartTime, CheckFinishTime, CheckTime),
		person(Pers, CheckTime)),
	vehicleNotMoving(Veh, JoinTime).

% Считается, что человек в определенный момент находится на средстве передвижения, если это мотоцикл,
% велосипед и т. п., а также если положение 2d обрамляющей рамки человека и средства передвижения
% соотносятся определенным образом.
onVehic(Veh, Pers, Time) :-
	(vehicle(Veh, motorbike, Time); vehicle(Veh, bicycle, Time)),
	bounds(Veh, BoxVeh, Time),
	person(Pers, Time),
	bounds(Pers, BoxPers, Time),
	boxIntersection(BoxVeh, BoxPers, BoxInt),
	area(BoxVeh, VehArea),
	area(BoxPers, PersArea),
	area(BoxInt, IntersecArea),
	IntersecArea/PersArea > 0.4,
	IntersecArea/VehArea > 0.3.

% Возвращает обрамляющую рамку, полученную пересечением двух других. Если рамки не пересекаются, тогда либо XInt1 будет больше XInt2,
% либо YInt1 будет больше YInt2, либо оба.
boxIntersection(box(Xf1, Yf1, Xf2, Yf2), box(Xs1, Ys1, Xs2, Ys2), box(XInt1, YInt1, XInt2, YInt2)) :-
	XInt1 is max(Xf1, Xs1),
	YInt1 is max(Yf1, Ys1),
	XInt2 is min(Xf2, Xs2),
	YInt2 is min(Yf2, Ys2).

% Вычисляет площадь обрамляющей рамки. X1, Y1, X2, Y2 должны быть заданы.
area(box(X1, Y1, X2, Y2), Area) :-
	X1 < X2,
	Y1 < Y2,
	Area is (X2 - X1) * (Y2 - Y1),
	!.
area(box(_, _, _, _), 0). 

% Считаем, что Actor1 подошел к Actor2, если начальное расстояние между людьми было более 2м, а стало не более 1.5м, 
% и бОльшую часть пути прошел Actor1. Либо если начальное расстояния было не более 2м, при этом считаем, что
% FinishTime равно StartTime.
% Следует всегда задавать StartTime, поскольку иначе предикат может просто подобрать start time таким образом, что
% начальное и конечное расстояния будут менее 2м.
approached(Actor1, Actor2, MaxDur, StartTime, FinishTime) :- 
	% Начальное время должно быть задано, иначе данный предикат будет подбирать начальное время, чтобы расстояние между
	% людьми было мало.
	number(StartTime),
	distanceGreater(Actor1, Actor2, dist(2, m), StartTime),
	distanceNotGreaterFixedStartTime(Actor1, Actor2, MaxDur, dist(1.5, m), StartTime, FinishTime),
	distance(Actor1, StartTime, FinishTime, Dist1),
	distance(Actor2, StartTime, FinishTime, Dist2),
	Dist1 > Dist2,
	!.
approached(Actor1, Actor2, _MaxDur, StartTime, StartTime) :- 
	number(StartTime),
	distanceNotGreater(Actor1, Actor2, dist(2, m), StartTime).

distanceNotGreaterFixedStartTime(Actor1, Actor2, MaxInt, MaxDist, StartTime, FinishTime) :-
	number(StartTime),
	timeGreater(StartTime, MaxInt, FinishTime),
	distanceNotGreater(Actor1, Actor2, MaxDist, FinishTime),
	!.

% Расстояние, которое прошел Actor за указанный период времени.
distance(Actor, StartTime, FinishTime, Dist) :-
	position(Actor, _, StartTime),
	position(Actor, _, FinishTime),
	StartTime =< FinishTime,
	distanceAcc(Actor, StartTime, FinishTime, Dist).

distanceAcc(Actor, StartTime, FinishTime, DistAcc) :-
	StartTime < FinishTime,
	holdsAtNearestPrevTime(position(Actor, P1), FinishTime, PrevFinishTime),
	position(Actor, P2, FinishTime),
	distanceAcc(Actor, StartTime, PrevFinishTime, PrevDistAcc),
	distanceBetweenPoints(P1, P2, Dist),
	% Тут погрешность учитывать не следует, так как нам нужно выбрать одного из двух людей.
	DistAcc is PrevDistAcc + Dist,
	!.
distanceAcc(_Actor, _StartTime, _FinishTime, 0).

% Считаем, что Pers1 взял вещь у Pers2 в случае, если произошло действие передачи предмета, либо сумка перешла
% от Pers2 во владение Pers1.
tookAnItem(Pers1, Pers2, StartTime, FinishTime) :-
	(holds(transferObject(Pers1, Pers2), StartTime, FinishTime) ;
	possesses(Pers2, Bag, StartTime),
	possesses(Pers1, Bag, FinishTime),
	StartTime < FinishTime,
	Pers1 \= Pers2,
	\+ (posesses(_, Bag, Time), StartTime < Time, Time < FinishTime)).

% Считаем, что человек владеет сумкой, если обрамляющая рамка сумки имеет пересечение только с рамкой этого человека.
% Если же пересечение есть более чем с одним человеком, считаем, что владелец среди них тот, кто владел сумкой последним.
% Если же последний владелец не имеет пересечения с сумкой, считаем, что текущим владельцем является человек с наибольшим
% пересечением.
possesses(Pers, Bag, Time) :-
	bag(Bag, Time),
	bounds(Bag, BoxBag, Time),
	intersectPerson(BoxBag, Pers, _, Time),
	\+ (intersectPerson(BoxBag, AnotherPers, _, Time), AnotherPers \= Pers),
	!.
possesses(Pers, Bag, Time) :-
	bag(Bag, Time),
	bounds(Bag, BoxBag, Time),
	intersectPerson(BoxBag, Pers, _, Time),
	holdsAtNearestPrevTime(bag(Bag), Time, PrevTime),
	possesses(Pers, Bag, PrevTime),
	!.
possesses(Pers, Bag, Time) :-
	bag(Bag, Time),
	bounds(Bag, BoxBag, Time),
	intersectPerson(BoxBag, Pers, IntersectAreaPers, Time),
	\+ (intersectPerson(BoxBag, AnotherPers, IntersectAreaAnotherPers, Time),
		AnotherPers \= Pers,
		IntersectAreaAnotherPers > IntersectAreaPers),
	!.

intersectPerson(Box, Pers, IntersectArea, Time) :-
	person(Pers, Time),
	bounds(Pers, BoxPers, Time),
	boxIntersection(Box, BoxPers, BoxInt),
	area(BoxInt, IntersectArea),
	IntersectArea > 0.

% DeviantBehaviour
robbingWithVehicle(Vict, Robb, Veh, StartTime, FinishTime) :-
	droveUpTo(Veh, Vict, StartTime, ArriveTime),
	leftAVehicle(Veh, Robb, ArriveTime),
	approached(Robb, Vict, dur(30, sec), ArriveTime, ApproachTime),
	(tookAnItem(Robb, Vict, ActStartTime, ActFinishTime) ;
		holds(searchedPockets(Robb, Vict), ActStartTime, ActFinishTime) ;
		holds(tugBetween(Robb, Vict), ActStartTime, ActFinishTime)),
	ApproachTime =< ActStartTime,
	incTime(ActFinishTime, dur(1, min), RetreatTimeLimit),
	jointAVehicle(Veh, Robb, RetreatTime),
	ActFinishTime =< RetreatTime,
	RetreatTime =< RetreatTimeLimit,
	droveAway(Veh, Vict, RetreatTime, FinishTime).

% Определяет направление, в котором объект покинул сцену (человек, транспорт и др.). Для одной камеры может быть задано
% множество направлений, которые определяются положением на кадре и вектором. Положения нескольких направлений могут пересекаться.
% Считается, что объект покинул сцену в определенном напрвлении, если на данный момент объект отсутствует на сцене,
% в последний момент присутствия объект находился в области этого направления, и вектор движения объекта соответствовал
% вектору этого направления (скалярное произведение векторов было больше 0). В случае, если объект находился в области
% нескольких пересекающихся направлений, выбирается то направление, с которым скалярное произведение векторов наибольшее.
% Ожидается, что Obj будет задан, а LeavingTime - нет.
leavingDirection(Obj, presents, LeavingTime) :-
	maxTimeSlice(CurrentTime),
	position(Obj, _, CurrentTime),
	LeavingTime is CurrentTime,
	!.
leavingDirection(Obj, Direction, LeavingTime) :-
	maxTimeSlice(CurrentTime),
	holdsAtNearestPrevTime(position(Obj, point(X, Y, _Z)), CurrentTime, LeavingTime),
	movementVector(Obj, LeavingTime, vector(MVx, MVy, _MVz)),
	directionForMovement(Direction, X, Y, MVx, MVy, DirectCoorientation),
    \+ (directionForMovement(AnotherDirection, X, Y, MVx, MVy, AnotherDirectCoorientation),
    AnotherDirection \= Direction,
    AnotherDirectCoorientation > DirectCoorientation),
    !.

directionForMovement(Direction, X, Y, Vx, Vy, Coorientation) :-
	areaLocation(AreaLocation, X, Y),
    direction(Direction, AreaLocation, DVx, DVy),
    cosBetween(vector(Vx, Vy, 0), vector(DVx, DVy, 0), Coorientation),
    Coorientation > 0.

% Определяет направление движения объекта в определенный момент. В качестве направления движения принимается вектор
% смещения между двумя точками во времени. Выбирается ближайшая по времени начальная точка таким образом, чтобы размер
% вектора был не менее 1 метра. Если смещение объекта за последнюю минуту меньше 1 метра, то считается, что движения нет.
movementVector(Obj, FinishTime, V) :-
	position(Obj, P2, FinishTime),
	timeSmaller(FinishTime, dur(1, min), StartTime),
	position(Obj, P1, StartTime),
	vectorOf(P1, P2, V),
	vectorLength(V, Dist),
	Dist >= 1000,
	!.

% Считаем, что человек смотрит по направлению к предмету, если угол между вектором направления головы человека и вектором
% от человека до предмета был не более 30 градусов. При этом рассматриваются векторы, проецируемые на поверхность.
% Положение объекта может быть задано в виде точки, обрамляющей фигуры или линии, спроецированной на поверхность.
looksToward(Pers, Obj, Time) :-
	person(Pers, Time),
	(person(Obj, Time) ; animal(Obj, _, Time)),
	lookDirection(Pers, LookV, Time),
	position(Pers, PersP, Time),
	position(Obj, ObjP, Time),
	vectorOf(PersP, ObjP, PosV),
	cosBetween(PosV, LookV, Cos),
	acos(Cos) =< 30 * pi / 180.
looksToward(Pers, Obj, Time) :-
	person(Pers, Time),
	(vehicle(Obj, _, Time) ; window(Obj, Time) ; door(Obj, Time) ; fence(Obj, Time) ; wall(Obj, Time)),
	lookDirection(Person, LookV, Time),
	position(Person, PersP, Time),
	volumetricPosition(Obj, ObjShape, Time),
	angleBetweenVectorAndShape(LookV, PersP, ObjShape, 1_000_000, AngleRad),
	AngleRad =< 30 * pi / 180.

% Находит минимальный угол между заданным вектором и вектором, исходящим от заданной точки P до фигуры,
% описанной массивом точек.
% Ожидается, что фигура будет описана как минимум двумя точками.
angleBetweenVectorAndShape(V, P, [P1, P2 | TP], AngleRadAcc, AngleRad) :-
	vectorOf(P, P1, A),
	vectorOf(P, P2, C),
	B = V,
	vectorMult(A, B, AxB),
	vectorMult(A, C, AxC),
	vectorMult(C, B, CxB),
	vectorMult(C, A, CxA),
	scalarMult(AxB, AxC, S1), scalarMult(CxB, CxA, S2),
	((S1 >= 0, S2 >= 0) -> AngleRad is 0;
		cosBetween(A, B, ABCos), cosBetween(C, B, CBCos),
		NewAngleRadAcc is min(AngleRadAcc, min(acos(ABCos), acos(CBCos))),
		angleBetweenVectorAndShape(V, P, [P2 | TP], NewAngleRadAcc, AngleRad)).
angleBetweenVectorAndShape(_V, _P, [_], AngleRadAcc, AngleRadAcc).

looksTowardHappensBetween(Pers, Obj, StartTime, FinishTime) :-
	timeBetween(StartTime, FinishTime, Time),
    looksToward(Pers, Obj, Time),
    !.

hittingAnObject(Pers, Obj, MaxDistance, StartTime, FinishTime) :-
	holds(hitting(Pers), StartTime, FinishTime),
	distanceNotGreaterHappensBetween(Pers, Obj, MaxDistance, StartTime, FinishTime),
	Pers \= Obj,
	looksTowardHappensBetween(Pers, Obj, StartTime, FinishTime).

throwingAtObject(Pers, Obj, MaxDistance, StartTime, FinishTime) :-
	holds(throwing(Pers), StartTime, FinishTime),
	distanceNotGreater(Pers, Obj, MaxDistance, FinishTime),
	Pers \= Obj,
	looksTowardHappensBetween(Pers, Obj, StartTime, FinishTime).
	

% DeviantBehaviour
% Определяет нанесение удара человеком животному.
hittingAnAnimal(Pers, Animal, StartTime, FinishTime) :-
	animal(Animal, _, StartTime),
	hittingAnObject(Pers, Animal, dist(1, m), StartTime, FinishTime).

% DeviantBehaviour
% Определяет нанесение удара человеком другому человеку.
hittingAPerson(Pers, Vict, StartTime, FinishTime) :-
	person(Vict, StartTime),
	hittingAnObject(Pers, Vict, dist(1.5, m), StartTime, FinishTime).

% DeviantBehaviour
% Расцениваем удар по двери как попытку взлома.
breakingInto(Pers, Door, StartTime, FinishTime) :-
	door(Door, StartTime),
	hittingAnObject(Pers, Door, dist(1, m), StartTime, FinishTime).

% DeviantBehaviour
% Удар по машине или сильный бросок в сторону машины с близкого расстояния расцениваем как вандализм.
vandalisingAVehicle(Pers, Veh, StartTime, FinishTime) :-
	vehicle(Veh, _, StartTime),
	hittingAnObject(Pers, Veh, dist(1, m), StartTime, FinishTime).
vandalisingAVehicle(Pers, Veh, StartTime, FinishTime) :-
	vehicle(Veh, _, StartTime),
	throwingAtObject(Pers, Veh, dist(5, m), StartTime, FinishTime).

% DeviantBehaviour
% Удар по окну или сильный бросок в сторону окна с близкого расстояния расцениваем как вандализм.
vandalisingAWindow(Pers, Wind, StartTime, FinishTime) :-
	window(Wind, StartTime),
	hittingAnObject(Pers, Wind, dist(1, m), StartTime, FinishTime).
vandalisingAWindow(Pers, Wind, StartTime, FinishTime) :-
	window(Wind, StartTime),
	throwingAtObject(Pers, Wind, dist(5, m), StartTime, FinishTime).

% DeviantBehaviour
% Предполагаем, что человек намерен украсть что-то из машины, если он останавливался у, по крайней мере, 3х машин
% и наклонялся, чтобы посмотреть в окно.
stealingFromACar(Pers, Car1, Car2, Car3, StartTime, FinishTime) :-
	lookingIntoACar(Pers, Car1, StartTime1, FinishTime1),
	lookingIntoACar(Pers, Car2, StartTime2, FinishTime2),
	lookingIntoACar(Pers, Car3, StartTime3, FinishTime3),
	Car1 \= Car2,
	Car1 \= Car3,
	Car2 \= Car3,
	StartTime is min(min(StartTime1, StartTime2), StartTime3),
	FinishTime is max(max(FinishTime1, FinishTime2), FinishTime3),
	FinishTime - StartTime =< 600_000.

% Описывает факт того, что человек смотрит в салон машины.
lookingIntoACar(Pers, Car, StartTime, FinishTime) :-
	performingActionTowardACar(Pers, bendingForward(Pers), Car, StartTime, FinishTime).

performingActionTowardACar(Pers, Action, Car, StartTime, FinishTime) :-
	holds(Action, StartTime, ActionFinishTime),
	person(Pers, StartTime),
	vehicle(Car, car, StartTime),
	distanceNotGreater(Pers, Car, dist(1, m), StartTime),
	FinishTime is StartTime + 2000,
	FinishTime =< ActionFinishTime,
	movedLessThen(Pers, dist(1, m), StartTime, FinishTime),
	looksTowardMostOfTime(Pers, Car, StartTime, FinishTime).

% Проверяет, что объект передвинулся менее чем на MaxDist между StartTime и FinishTime.
movedLessThen(Obj, MaxDist, StartTime, FinishTime) :-
	toMm(MaxDist, MaxDistMm),
	position(Obj, P1, StartTime),
	holdsAtNearestPrevTime(position(Obj, P2), FinishTime, _),
	distanceBetweenPoints(P1, P2, Dist),
	Dist =< MaxDistMm.

% Человек смотрел по направлению к объекту большу часть времени между StartTime и FinishTime.
looksTowardMostOfTime(Pers, Obj, StartTime, FinishTime) :-
	looksTowardMostOfTimeAcc(Pers, Obj, StartTime, FinishTime, 0, 0).

looksTowardMostOfTimeAcc(_Pers, _Obj, StartTime, FinishTime, LookAcc, VisibleAcc) :-
	StartTime > FinishTime,
	!,
	LookAcc / max(1, VisibleAcc) >= 0.5.
looksTowardMostOfTimeAcc(Pers, Obj, StartTime, FinishTime, LookAcc, VisibleAcc) :-
	person(Pers, StartTime),
	position(Obj, _, StartTime),
	!,
	NewVisibleAcc is VisibleAcc + 1,
	(looksToward(Pers, Obj, StartTime) -> NewLookAcc is LookAcc + 1; NewLookAcc is LookAcc),
	timeStep(TimeStep),
	NextStartTime is StartTime + TimeStep,
	looksTowardMostOfTimeAcc(Pers, Obj, NextStartTime, FinishTime, NewLookAcc, NewVisibleAcc).
looksTowardMostOfTimeAcc(Pers, Obj, StartTime, FinishTime, LookAcc, VisibleAcc) :-
	timeStep(TimeStep),
	NextStartTime is StartTime + TimeStep,
	looksTowardMostOfTimeAcc(Pers, Obj, NextStartTime, FinishTime, LookAcc, VisibleAcc).

% DeviantBehaviour
% Предполагаем, что человек может протыкать шины, если он останавливался у по крайней мере 3х машин
% и приседал.
cuttingCarWheels(Pers, Car1, Car2, Car3) :-
	squatingInFrontOfACar(Pers, Car1, StartTime1, FinishTime1),
	squatingInFrontOfACar(Pers, Car2, StartTime2, FinishTime2),
	squatingInFrontOfACar(Pers, Car3, StartTime3, FinishTime3),
	Car1 \= Car2,
	StartTime is min(min(StartTime1, StartTime2), StartTime3),
	FinishTime is max(max(FinishTime1, FinishTime2), FinishTime3),
	FinishTime - StartTime =< 600_000.

squatingInFrontOfACar(Pers, Car, StartTime, FinishTime) :-
	performingActionTowardACar(Pers, squating(Pers), Car, StartTime, FinishTime).

% DeviantBehaviour
% Предполагаем, что произошла стрельба, если человек упал, и непосредственно перед падением (в течение нескольких секунд) был человек,
% принимающий позу выстрела и смотрящий в сторону падающего.
shootingAPerson(Shooter, Vict, StartTime, FinishTime) :-
	holds(falling(Vict), FallingTime, FinishTime),
	shootTowardRecently(Shooter, Vict, FallingTime, StartTime).

% Данное определение в отдельности имеет низкую точность детектирования (большой false positive).
shootTowardRecently(Shooter, Obj, CurrentTime, Time) :-
	MinRecentTime is max(0, CurrentTime - 5_000),
	timeBetween(MinRecentTime, CurrentTime, Time),
	holdsAt(shootingPose(Shooter), Time),
	Shooter \= Obj,
	looksToward(Shooter, Obj, Time).

% DeviantBehaviour
% Предполагаем, что происходит стрельба, если человек перемещается боком или спиной в позе выстрела. Рассматривается
% расстояние перемещения - 5 метров за 15 секунд.
shooting(Shooter, StartTime, FinishTime) :-
	possibleShootingPeriod(Shooter, StartTime, FinishTime),
	shootingBetween(Shooter, StartTime, FinishTime).

possibleShootingPeriod(Shooter, StartTime, FinishTime) :-
	holds(shootingPose(Shooter), PoseStartTime, PoseFinishTime),
	timeBetween(PoseStartTime, PoseFinishTime, StartTime),
	FinishTime is StartTime + 15_000.

shootingBetween(Shooter, StartTime, FinishTime) :-
	shootingBetweenAcc(Shooter, StartTime, FinishTime, 0).

shootingBetweenAcc(_Shooter, StartTime, FinishTime, MovementDistAcc) :-
	StartTime >= FinishTime,
	MovementDistAcc >= 5_000.
shootingBetweenAcc(Shooter, StartTime, FinishTime, MovementDistAcc) :-
	StartTime < FinishTime,
	timeStep(Step),
	% Возможно, следует взять более удаленную точку для проверки вектора движения (например, 1 сек), чтобы вектор
	% был более робастым
	NextTime is StartTime + Step,
	(holdsAt(shootingPose(Shooter), StartTime), movedBackwardsOrSideways(Shooter, StartTime, NextTime, MovementDist) ->
		NewMovementDistAcc is MovementDistAcc + MovementDist ;
		NewMovementDistAcc is MovementDistAcc),
	shootingBetweenAcc(Shooter, NextTime, FinishTime, NewMovementDistAcc).

% Человек передвинулся спиной или боком, если угол между вектором смещения за определенный период времени и вектором взгляда
% при старте не меньше 60 градусов.
movedBackwardsOrSideways(Pers, StartTime, FinishTime, MovementDist) :-
	position(Pers, P1, StartTime),
	position(Pers, P2, FinishTime),
	lookDirection(Pers, LookV, StartTime),
	vectorOf(P1, P2, MovementV),
	cosBetween(MovementV, LookV, Cos),
	acos(Cos) >= 60 * pi / 180,
	distanceBetweenPoints(P1, P2, MovementDist).

% DeviantBehaviour
% Предполагаем, что происходит драка, если два человека быстро перемещаются в одном направлении (один отступает,
% а другой наступает), смотря друг на друга и находясь на расстоянии не более 3 метров друг от друга. И во время
% перемещения хотя бы один из людей наносит удар, либо принимает offensive позу (руки подняты, как в боксе),
% defensive позу (руки подняты на уровне груди для защиты), либо surrender позу (с поднятыми руками).
% Детектируется совместное перемещение на расстояние 5 метров в течение 10 секунд.
fighting(Pers1, Pers2, StartTime, FinishTime) :-
	(hittingAPerson(Pers1, Pers2, SuspiciousActonStartTime, SuspiciousActonFinishTime) ;
	holds(defensivePose(Pers1), SuspiciousActonStartTime, SuspiciousActonFinishTime) ;
	holds(offensivePose(Pers1), SuspiciousActonStartTime, SuspiciousActonFinishTime) ;
	holds(surrenderPose(Pers1), SuspiciousActonStartTime, SuspiciousActonFinishTime)),
	possibleFightinglPeriod(SuspiciousActonStartTime, SuspiciousActonFinishTime, StartTime, FinishTime),
	movingJointly(Pers1, Pers2, dist(5, m), StartTime, FinishTime).

possibleFightinglPeriod(SuspiciousActonStartTime, SuspiciousActonFinishTime, StartTime, FinishTime) :-
	STime is max(0, SuspiciousActonStartTime - 10_000),
	FTime is SuspiciousActonFinishTime,
	timeBetween(STime, FTime, StartTime),
	FinishTime is StartTime + 10_000.

movingJointly(Pers1, Pers2, MinDistance, StartTime, FinishTime) :-
	toMm(MinDistance, MinDistanceMm),
	movingJointlyAcc(Pers1, Pers2, MinDistanceMm, StartTime, FinishTime, 0).

movingJointlyAcc(_Pers1, _Pers2, MinDistanceMm, StartTime, FinishTime, MovementDistAcc) :-
	StartTime >= FinishTime,
	MovementDistAcc >= MinDistanceMm.
movingJointlyAcc(Pers1, Pers2, StartTime, FinishTime, MovementDistAcc) :-
	StartTime < FinishTime,
	timeStep(Step),
	NextTime is StartTime + Step,
	(movedJointly(Pers1, Pers2, StartTime, NextTime, MovementDist) ->
		NewMovementDistAcc is MovementDistAcc + MovementDist ;
		NewMovementDistAcc is MovementDistAcc),
	movingJointlyAcc(Pers1, Pers2, NextTime, FinishTime, NewMovementDistAcc).

movedJointly(Pers1, Pers2, StartTime, FinishTime, MovementDist) :-
	peopleNearAndLookAtEachOther(Pers1, Pers2, 3_000, StartTime),
	peopleNearAndLookAtEachOther(Pers1, Pers2, 3_000, FinishTime),
	position(Pers1, P1StartPos, StartTime),
	position(Pers2, P2StartPos, StartTime),
	pointBetween(P1StartPos, P2StartPos, PeopleStartPos),
	position(Pers1, P1FinishPos, FinishTime),
	position(Pers2, P2FinishPos, FinishTime),
	pointBetween(P1FinishPos, P2FinishPos, PeopleFinishPos),
	distanceBetweenPoints(PeopleStartPos, PeopleFinishPos, MovementDist).
	
peopleNearAndLookAtEachOther(Pers1, Pers2, MaxDistMm, Time) :-
	peopleNear(Pers1, Pers2, MaxDistMm, Time),
	looksToward(Pers1, Pers2, Time),
	looksToward(Pers2, Pers1, Time).

peopleNear(Pers1, Pers2, MaxDistMm, Time) :-
	person(Pers1, Time),
	person(Pers2, Time),
	position(Pers1, P1, Time),
	position(Pers2, P2, Time),
	Pers1 \= Pers2,
	distanceBetweenPoints(P1, P2, Dist),
	% todo: учесть погрешность
	Dist =< MaxDistMm.

% DeviantBehaviour
% Если ребенок находится на проезжей части и нет взрослого, который находился бы поблизости него (в 3х метрах) в течение
% последних 3х секунд.
unattendedСhildOnAHighway(Child, StartTime, FinishTime) :-
	person(Child, FinishTime),
	child(Child),
	onAHighway(Child, FinishTime),
	StartTime is FinishTime - 3_000,
	timeSlice(StartTime),
	\+ (timeBetween(StartTime, FinishTime, Time),
		peopleNear(Child, Adult, 3_000, Time),
		\+ (child(Adult))).

onAHighway(Pers, Time) :-
	position(Pers, point(X, Y, _Z), Time),
	areaLocation(Area, X, Y),
	highway(_Highway, Area).


% DeviantBehaviour
% Ограбление, при котором группа людей (от двух и более) подходит к человеку, и в течение некоторого
% времени (от 10 сек) располагается рядом с человеком, обыскивает, забирает предмет, или происходит перетягивание. После чего
% группа удаляется, либо человек сбегает.
robbingByGroupOfStrangers(Vict, group(Rob1, Rob2), StartTime, FinishTime) :-
	group(Rob1, Rob2, StartTime),
	groupDistanceGreater(group(Rob1, Rob2), Vict, dist(5, m), StartTime),
	groupApproached(group(Rob1, Rob2), Vict, dur(10, sec), StartTime, ApproachTime),
	(tookAnItem(Rob1, Vict, ActStartTime, ActFinishTime) ;
		holds(searchedPockets(Rob1, Vict), ActStartTime, ActFinishTime) ;
		holds(tugBetween(Rob1, Vict), ActStartTime, ActFinishTime)),
	ApproachTime =< ActStartTime,
	groupDistanceNotGreater(group(Rob1, Rob2), Vict, dist(1.5, m), EncounterTime),
	toMm(dur(10, sec), MinEncounterDurMm),
	EncounterTime - ApproachTime >= MinEncounterDurMm,
	incTime(ActFinishTime, dur(1, min), RetreatTimeLimit),
	% Либо группа скрылась, либо человек убежал
	groupDistanceGreater(group(Rob1, Rob2), Vict, dist(5, m), RetreatTime),
	EncounterTime =< RetreatTime,
	ActFinishTime =< RetreatTime,
	RetreatTime =< RetreatTimeLimit,
	FinishTime is RetreatTime.

% Пара людей считаются группой, если в течение 5 секунд расстояние между ними было не более 1.5 метров.
group(Pers1, Pers2, FinishTime) :-
	timeSlice(FinishTime),
	person(Pers1, FinishTime),
	person(Pers2, FinishTime),
	Pers1 \= Pers2,
	decTime(FinishTime, dur(5, sec), MaxStartTime),
	holdsAtNearestPrevTime(person(Pers1), MaxStartTime, StartTime),
	holdsAtNearestPrevTime(person(Pers2), MaxStartTime, StartTime),
	\+ (timeBetween(StartTime, FinishTime, Time),
		distanceGreater(Pers1, Pers2, dist(1.5, m), Time)).
% Определяет, что отношение группы транзитивно. Можно также добавить, что два человека относятся к одной группе,
% если они относились к группе когда-либо. Тут, конечно, возможен false positive, например, если люди просто стоят на светофоре.
group(Pers1, Pers2, FinishTime) :-
	group(Pers1, Pers3, FinishTime),
	group(Pers2, Pers3, FinishTime),
	Pers1 \= Pers2.

% Проверяет, что для обоих членов группы расстояние между человеком до объекта более указанного значения.
groupDistanceGreater(group(Pers1, Pers2), Obj, MinDist, Time) :-
	distanceGreater(Pers1, Obj, MinDist, Time),
	distanceGreater(Pers2, Obj, MinDist, Time).

groupDistanceNotGreater(group(Pers1, Pers2), Obj, MaxDist, Time) :-
	distanceNotGreater(Pers1, Obj, MaxDist, Time),
	distanceNotGreater(Pers2, Obj, MaxDist, Time).

% Считается, что группа подошла к человеку, если оба члена группы подошли к нему.
groupApproached(group(Pers1, Pers2), Pers3, MaxDur, StartTime, FinishTime) :-
	approached(Pers1, Pers3, MaxDur, StartTime, FinishTime1),
	approached(Pers2, Pers3, MaxDur, StartTime, FinishTime2),
	FinishTime is max(FinishTime1, FinishTime2).

% DeviantBehaviour
% Человек ехал на мотоцикле или велосипеде, к нему подошел грабитель, после чего грабитель уехал на средстве передвижения,
% оставив человека позади. При этом средство передвижение могли либо отобрать, либо человек мог припарковаться и отойти.
hijackingABike(Vict, Veh, Robb, StartTime, FinishTime) :-
	droveAtLeast(Vict, Veh, dist(5, m), dur(15, sec), StartTime, VictimDroveUpTime),
	person(Robb, RobberInTheDistanceTime),
	distanceGreater(Robb, Veh, dist(5, m), RobberInTheDistanceTime),
	StartTime =< RobberInTheDistanceTime,
	distanceNotGreater(Robb, Veh, dist(1.5, m), RobbApproachedTime),
	RobberInTheDistanceTime =< RobbApproachedTime,
	VictimDroveUpTime =< RobbApproachedTime,
	droveAtLeast(Robb, Veh, dist(5, m), RetreatStartTime, RetreatFinishTime),
	RobbApproachedTime =< RetreatStartTime,
	distanceGreater(Vict, Veh, dist(5, m), VictimInTheDistanceTime),
	RetreatStartTime =< VictimInTheDistanceTime,
	FinishTime is max(RetreatFinishTime, VictimInTheDistanceTime).

% Человек проехал на средстве передвижения как минимум MinDist за как максимум MaxDur. Считаем, что оперделенный человек
% проехал на средстве передвижения, если в начальный и конечный момент движения он находился на средстве передвижения.
droveAtLeast(Pers, Veh, MinDist, MaxDur, StartTime, FinishTime) :-
	(vehicle(Veh, motorbike, FinishTime) ; vehicle(Veh, bicycle, FinishTime)),
	moved(Veh, MinDist, MaxDur, StartTime, FinishTime),
	onVehic(Veh, Pers, StartTime),
	onVehic(Veh, Pers, FinishTime).

% DeviantBehaviour
% Как только человек вышел из машины, к нему подошел грабитель, отобрал ключи, сел в машину и уехал, оставив человека позади.
hijackingACar(Vict, Car, Robb, StartTime, FinishTime) :-
	vehicle(Car, car, StartTime),
	leftAVehicle(Car, Vict, StartTime),
	% На случай, если просто два человека вышли из одной машины чтобы попрощаться, можно было бы детектировать,
	% что грабитель стоял далеко от машины.
	approached(Robb, Vict, dur(1, min), StartTime, ApproachTime),
	(tookAnItem(Robb, Vict, ActStartTime, ActFinishTime) ;
		holds(searchedPockets(Robb, Vict), ActStartTime, ActFinishTime) ;
		holds(tugBetween(Robb, Vict), ActStartTime, ActFinishTime)),
	ApproachTime =< ActStartTime,
	jointAVehicle(Car, Robb, RobberJointACarTime),
	ActFinishTime =< RobberJointACarTime,
	inctTime(RobberJointACarTime, dur(1, min), RetreatStartTimeLimit),
	moved(Car, dist(1, m), dur(30, sec), RetreatStartTime, RetreatFinishTime),
	RobberJointACarTime =< RetreatStartTime,
	RetreatStartTime =< RetreatStartTimeLimit,
	distanceGreater(Vict, Car, dist(5, m), VictimInTheDistanceTime),
	RetreatStartTime =< VictimInTheDistanceTime,
	FinishTime is max(RetreatFinishTime, VictimInTheDistanceTime).